'use client'

import { useEffect, useState, useRef } from 'react'
import { useParams } from 'next/navigation'
import { supabase } from '@/lib/supabase'
import { Pago, Cliente, Suscripcion } from '@/lib/types'
import { format } from 'date-fns'
import jsPDF from 'jspdf'
import html2canvas from 'html2canvas'
import Image from 'next/image'

export default function ComprobantePublicoPage() {
  const params = useParams()
  const [pago, setPago] = useState<Pago | null>(null)
  const [cliente, setCliente] = useState<Cliente | null>(null)
  const [suscripcion, setSuscripcion] = useState<Suscripcion | null>(null)
  const [loading, setLoading] = useState(true)
  const [generandoPDF, setGenerandoPDF] = useState(false)
  const comprobanteRef = useRef<HTMLDivElement>(null)

  useEffect(() => {
    if (params.id) {
      fetchPago(params.id as string)
    }
  }, [params.id])

  async function fetchPago(id: string) {
    try {
      const { data, error } = await supabase
        .from('pagos')
        .select(`
          *,
          cliente:clientes(*),
          suscripcion:suscripciones(*)
        `)
        .eq('id', id)
        .single()

      if (error) throw error
      setPago(data)
      setCliente(data.cliente as Cliente)
      setSuscripcion(data.suscripcion as Suscripcion | null)
    } catch (error) {
      console.error('Error fetching pago:', error)
    } finally {
      setLoading(false)
    }
  }

  async function descargarPDF() {
    if (!comprobanteRef.current || !pago || !cliente) return

    setGenerandoPDF(true)

    try {
      const canvas = await html2canvas(comprobanteRef.current, {
        scale: 2,
        useCORS: true,
        logging: false,
      })

      const imgData = canvas.toDataURL('image/png')
      const pdf = new jsPDF('p', 'mm', 'a4')
      const pdfWidth = pdf.internal.pageSize.getWidth()
      const pdfHeight = pdf.internal.pageSize.getHeight()
      const imgWidth = canvas.width
      const imgHeight = canvas.height
      const ratio = Math.min(pdfWidth / imgWidth, pdfHeight / imgHeight)
      const imgX = (pdfWidth - imgWidth * ratio) / 2
      const imgY = 0

      pdf.addImage(imgData, 'PNG', imgX, imgY, imgWidth * ratio, imgHeight * ratio)
      pdf.save(`comprobante-pago-${pago.id.slice(0, 8)}.pdf`)
    } catch (error) {
      console.error('Error generating PDF:', error)
      alert('Error al generar el PDF')
    } finally {
      setGenerandoPDF(false)
    }
  }

  if (loading) {
    return <div className="text-center p-8">Cargando comprobante...</div>
  }

  if (!pago || !cliente) {
    return (
      <div className="text-center p-8">
        <h1 className="text-2xl font-bold mb-4">Comprobante no encontrado</h1>
        <p className="text-gray-600">El comprobante que buscas no existe o no está disponible.</p>
      </div>
    )
  }

  return (
    <div className="min-h-screen bg-gray-100 py-8">
      <div className="max-w-4xl mx-auto px-4">
        <div className="bg-white p-6 rounded-lg shadow mb-4">
          <div className="flex justify-between items-center">
            <h1 className="text-2xl font-bold">Comprobante de Pago</h1>
            <button
              onClick={descargarPDF}
              disabled={generandoPDF}
              className="bg-green-600 text-white px-4 py-2 rounded hover:bg-green-700 disabled:opacity-50 transition"
            >
              {generandoPDF ? 'Generando...' : '📄 Descargar PDF'}
            </button>
          </div>
        </div>

        <div ref={comprobanteRef} className="bg-white border-2 border-gray-300 p-8 shadow-lg" style={{ minHeight: '400px' }}>
          <div className="text-center mb-8">
            <div className="flex justify-center items-center gap-4 mb-4">
              <Image
                src="/img/logo_navigps.png"
                alt="NAVIGPS Logo"
                width={150}
                height={75}
                className="object-contain"
              />
            </div>
            <h1 className="text-4xl font-bold text-gray-800 mb-2">NAVIGPS</h1>
            <p className="text-gray-600 text-lg">Comprobante de Pago</p>
          </div>

          <div className="border-t-2 border-b-2 border-gray-400 py-4 mb-6">
            <div className="flex justify-between items-center mb-2">
              <span className="text-gray-600 font-medium">Número de Comprobante:</span>
              <span className="font-bold text-gray-800 text-lg">#{pago.id.slice(0, 8).toUpperCase()}</span>
            </div>
            <div className="flex justify-between items-center">
              <span className="text-gray-600 font-medium">Fecha de Emisión:</span>
              <span className="font-semibold text-gray-800">{format(new Date(), 'dd/MM/yyyy HH:mm')}</span>
            </div>
          </div>

          <div className="mb-6">
            <h3 className="text-xl font-semibold text-gray-800 mb-3 border-b pb-2">Datos del Cliente</h3>
            <div className="space-y-2 text-gray-700">
              <p><span className="font-medium">Nombre:</span> {cliente.nombre || cliente.empresa || 'N/A'}</p>
              {cliente.empresa && <p><span className="font-medium">Empresa:</span> {cliente.empresa}</p>}
              {cliente.email && <p><span className="font-medium">Email:</span> {cliente.email}</p>}
              {cliente.telefono && <p><span className="font-medium">Teléfono:</span> {cliente.telefono}</p>}
              {cliente.rfc && <p><span className="font-medium">RUT:</span> {cliente.rfc}</p>}
            </div>
          </div>

          <div className="mb-6">
            <h3 className="text-xl font-semibold text-gray-800 mb-3 border-b pb-2">Detalles del Pago</h3>
            <div className="bg-gray-50 p-4 rounded space-y-2">
              <div className="flex justify-between items-center">
                <span className="text-gray-700 font-medium">Monto:</span>
                <span className="text-3xl font-bold text-gray-900">${pago.monto.toLocaleString()}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-700">Fecha de Pago:</span>
                <span className="font-semibold text-gray-800">{format(new Date(pago.fecha_pago), 'dd/MM/yyyy')}</span>
              </div>
              {pago.metodo_pago && (
                <div className="flex justify-between">
                  <span className="text-gray-700">Método de Pago:</span>
                  <span className="font-semibold text-gray-800 capitalize">{pago.metodo_pago}</span>
                </div>
              )}
              {pago.referencia && (
                <div className="flex justify-between">
                  <span className="text-gray-700">Referencia:</span>
                  <span className="font-semibold text-gray-800">{pago.referencia}</span>
                </div>
              )}
              <div className="flex justify-between">
                <span className="text-gray-700">Estado:</span>
                <span className={`font-semibold ${
                  pago.estado === 'completado' ? 'text-green-600' :
                  pago.estado === 'pendiente' ? 'text-yellow-600' :
                  'text-red-600'
                }`}>
                  {pago.estado === 'completado' ? 'Completado' :
                   pago.estado === 'pendiente' ? 'Pendiente' :
                   'Cancelado'}
                </span>
              </div>
              {suscripcion && (
                <div className="flex justify-between">
                  <span className="text-gray-700">Suscripción:</span>
                  <span className="font-semibold text-gray-800">
                    {suscripcion.tipo === 'anual' ? 'Anual' : 'Mensual'} - ${suscripcion.precio.toLocaleString()}
                  </span>
                </div>
              )}
            </div>
          </div>

          {pago.notas && (
            <div className="mb-6">
              <h3 className="text-xl font-semibold text-gray-800 mb-2 border-b pb-2">Notas</h3>
              <p className="text-gray-700">{pago.notas}</p>
            </div>
          )}

          <div className="border-t-2 border-gray-400 pt-4 mt-6">
            <p className="text-sm text-gray-600 text-center">
              Este es un comprobante generado automáticamente por el sistema NAVIGPS
            </p>
            <p className="text-xs text-gray-500 text-center mt-2">
              {typeof window !== 'undefined' && window.location.href}
            </p>
          </div>
        </div>
      </div>
    </div>
  )
}
